<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use yii\web\ForbiddenHttpException;
use app\models\Application;
use app\models\User;

class AdminController extends Controller
{
    public $layout = 'admin';

    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity->isAdmin();
                        }
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    throw new ForbiddenHttpException('Доступ запрещен. Только для администраторов.');
                }
            ],
        ];
    }

    public function actionIndex()
    {
        return $this->render('index');
    }

    public function actionApplications()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Application::find()
                ->joinWith('user')
                ->orderBy(['application.created_at' => SORT_DESC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('applications', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionView($id)
    {
        $application = Application::find()
            ->joinWith('user')
            ->where(['application.id' => $id])
            ->one();

        if (!$application) {
            throw new \yii\web\NotFoundHttpException('Заявка не найдена.');
        }

        if (Yii::$app->request->isPost) {
            $newStatus = Yii::$app->request->post('status');
            if (in_array($newStatus, array_keys(Application::getAdminStatuses()))) {
                $application->status = $newStatus;
                if ($application->save(false)) { // false - отключаем валидацию
                    Yii::$app->session->setFlash('success', 'Статус заявки успешно обновлен.');
                    return $this->refresh();
                } else {
                    Yii::$app->session->setFlash('error', 'Ошибка при обновлении статуса: ' . implode(', ', $application->getFirstErrors()));
                }
            } else {
                Yii::$app->session->setFlash('error', 'Неверный статус.');
            }
        }

        return $this->render('view', [
            'application' => $application,
        ]);
    }

    public function actionStatistics()
    {
        $totalApplications = Application::find()->count();
        $newApplications = Application::find()->where(['status' => Application::STATUS_NEW])->count();
        $acceptedApplications = Application::find()->where(['status' => Application::STATUS_ACCEPTED])->count();
        $errorApplications = Application::find()->where(['status' => Application::STATUS_ERROR])->count();

        $categoryStats = Application::find()
            ->select([
                'category',
                'COUNT(*) as count',
                'SUM(amount) as total_amount'
            ])
            ->groupBy('category')
            ->asArray()
            ->all();


        return $this->render('statistics', [
            'totalApplications' => $totalApplications,
            'newApplications' => $newApplications,
            'acceptedApplications' => $acceptedApplications,
            'errorApplications' => $errorApplications,
            'categoryStats' => $categoryStats,
        ]);
    }

}
