<?php

use yii\helpers\Html;
use yii\grid\GridView;
use yii\bootstrap5\Alert;

/** @var yii\web\View $this */
/** @var yii\data\ActiveDataProvider $dataProvider */

$this->title = 'Управление заявками';
$this->params['breadcrumbs'][] = $this->title;
?>

<div class="admin-applications">
    <div class="admin-header">
        <div class="header-content">
            <h1 class="page-title"><?= Html::encode($this->title) ?></h1>
            <p class="page-subtitle">Управление всеми заявками пользователей системы</p>
        </div>
        <div class="header-actions">
            <div class="stats-badge">
                <i class="fas fa-file-alt"></i>
                <span><?= $dataProvider->getTotalCount() ?> заявок</span>
            </div>
        </div>
    </div>

    <?php if (Yii::$app->session->hasFlash('success')): ?>
        <div class="alert-message success">
            <div class="alert-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="alert-content">
                <?= Yii::$app->session->getFlash('success') ?>
            </div>
        </div>
    <?php endif; ?>

    <div class="admin-card">
        <div class="card-header">
            <div class="card-title-section">
                <div class="card-icon">
                    <i class="fas fa-list"></i>
                </div>
                <div>
                    <h3 class="card-title">Список заявок</h3>
                    <p class="card-subtitle">Все переданные показания пользователей</p>
                </div>
            </div>
            <div class="card-filters">
                <div class="filter-item">
                    <i class="fas fa-filter"></i>
                    <span>Фильтры</span>
                </div>
            </div>
        </div>
        <div class="card-body">
            <?= GridView::widget([
                'dataProvider' => $dataProvider,
                'tableOptions' => ['class' => 'applications-table'],
                'options' => ['class' => 'table-responsive'],
                'layout' => "{items}\n{pager}",
                'columns' => [
                    [
                        'class' => 'yii\grid\SerialColumn',
                        'header' => '#',
                        'contentOptions' => ['class' => 'serial-column'],
                        'headerOptions' => ['class' => 'serial-header']
                    ],
                    [
                        'attribute' => 'user_id',
                        'value' => function($model) {
                            return $model->user->first_name . ' ' . $model->user->last_name;
                        },
                        'label' => 'Пользователь',
                        'contentOptions' => ['class' => 'user-column'],
                        'headerOptions' => ['class' => 'user-header'],
                        'content' => function($model) {
                            return '<div class="user-cell">
                                <div class="user-avatar">' .
                                strtoupper(substr($model->user->first_name, 0, 1) . substr($model->user->last_name, 0, 1)) .
                                '</div>
                                <div class="user-info">
                                    <span class="user-name">' . $model->user->first_name . ' ' . $model->user->last_name . '</span>
                                    <span class="user-email">' . $model->user->email . '</span>
                                </div>
                            </div>';
                        },
                        'format' => 'raw'
                    ],
                    [
                        'attribute' => 'date',
                        'format' => 'date',
                        'label' => 'Дата показаний',
                        'contentOptions' => ['class' => 'date-column'],
                        'headerOptions' => ['class' => 'date-header']
                    ],
                    [
                        'attribute' => 'category',
                        'value' => function($model) {
                            return $model->getCategoryName();
                        },
                        'label' => 'Категория',
                        'contentOptions' => ['class' => 'category-column'],
                        'headerOptions' => ['class' => 'category-header'],
                        'content' => function($model) {
                            $icons = [
                                'electricity' => 'fas fa-bolt',
                                'water' => 'fas fa-tint',
                                'gas' => 'fas fa-fire'
                            ];
                            $icon = $icons[$model->category] ?? 'fas fa-file-alt';
                            $colors = [
                                'electricity' => 'electricity',
                                'water' => 'water',
                                'gas' => 'gas'
                            ];
                            $color = $colors[$model->category] ?? 'default';
                            return '<div class="category-cell ' . $color . '">
                                <i class="' . $icon . '"></i>
                                <span>' . $model->getCategoryName() . '</span>
                            </div>';
                        },
                        'format' => 'raw'
                    ],
                    [
                        'attribute' => 'previous_readings',
                        'value' => function($model) {
                            return number_format($model->previous_readings, 2, ',', ' ');
                        },
                        'label' => 'Предыдущие',
                        'contentOptions' => ['class' => 'readings-column'],
                        'headerOptions' => ['class' => 'readings-header']
                    ],
                    [
                        'attribute' => 'current_readings',
                        'value' => function($model) {
                            return number_format($model->current_readings, 2, ',', ' ');
                        },
                        'label' => 'Текущие',
                        'contentOptions' => ['class' => 'readings-column current'],
                        'headerOptions' => ['class' => 'readings-header']
                    ],
                    [
                        'attribute' => 'consumption',
                        'value' => function($model) {
                            return number_format($model->consumption, 2, ',', ' ');
                        },
                        'label' => 'Расход',
                        'contentOptions' => ['class' => 'consumption-column'],
                        'headerOptions' => ['class' => 'consumption-header']
                    ],
                    [
                        'attribute' => 'amount',
                        'value' => function($model) {
                            return number_format($model->amount, 2, ',', ' ') . ' руб.';
                        },
                        'label' => 'Сумма',
                        'contentOptions' => ['class' => 'amount-column'],
                        'headerOptions' => ['class' => 'amount-header']
                    ],
                    [
                        'attribute' => 'status',
                        'value' => function($model) {
                            return '<span class="status-badge ' . $model->status . '">' . $model->getStatusName() . '</span>';
                        },
                        'format' => 'raw',
                        'label' => 'Статус',
                        'contentOptions' => ['class' => 'status-column'],
                        'headerOptions' => ['class' => 'status-header']
                    ],
                    [
                        'attribute' => 'created_at',
                        'format' => 'datetime',
                        'label' => 'Создана',
                        'contentOptions' => ['class' => 'created-column'],
                        'headerOptions' => ['class' => 'created-header']
                    ],
                    [
                        'class' => 'yii\grid\ActionColumn',
                        'template' => '{view}',
                        'contentOptions' => ['class' => 'actions-column'],
                        'headerOptions' => ['class' => 'actions-header'],
                        'buttons' => [
                            'view' => function ($url, $model, $key) {
                                return Html::a('<i class="fas fa-eye"></i> Просмотр', $url, [
                                    'class' => 'btn-view'
                                ]);
                            },
                        ],
                    ],
                ],
            ]); ?>
        </div>
    </div>

    <?php if (!$dataProvider->getCount()): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-file-alt"></i>
            </div>
            <h3>Заявок пока нет</h3>
            <p>Пользователи еще не создали заявки на передачу показаний</p>
        </div>
    <?php endif; ?>
</div>

<style>
    /* Main Container */
    .admin-applications {
        min-height: 100vh;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        padding: 2rem 1rem;
    }

    /* Header */
    .admin-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-end;
        margin-bottom: 2rem;
    }

    .header-content {
        flex: 1;
    }

    .page-title {
        font-size: 2.25rem;
        font-weight: 700;
        color: white;
        margin: 0 0 0.5rem 0;
    }

    .page-subtitle {
        color: rgba(255, 255, 255, 0.9);
        font-size: 1.1rem;
        margin: 0;
    }

    .header-actions {
        display: flex;
        gap: 1rem;
    }

    .stats-badge {
        background: rgba(255, 255, 255, 0.2);
        backdrop-filter: blur(10px);
        border: 1px solid rgba(255, 255, 255, 0.3);
        border-radius: 12px;
        padding: 0.75rem 1.25rem;
        color: white;
        display: flex;
        align-items: center;
        gap: 0.5rem;
        font-weight: 600;
    }

    .stats-badge i {
        font-size: 1.1rem;
    }

    /* Alert */
    .alert-message {
        display: flex;
        align-items: flex-start;
        gap: 1rem;
        padding: 1.25rem;
        border-radius: 12px;
        margin-bottom: 2rem;
        background: rgba(46, 204, 113, 0.1);
        border: 1px solid rgba(46, 204, 113, 0.3);
        color: #27ae60;
    }

    .alert-icon {
        font-size: 1.25rem;
        margin-top: 0.125rem;
    }

    .alert-content {
        flex: 1;
    }

    /* Card */
    .admin-card {
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(20px);
        border-radius: 20px;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
        overflow: hidden;
        margin-bottom: 2rem;
    }

    .card-header {
        padding: 1.5rem 2rem;
        border-bottom: 1px solid #e8eeff;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .card-title-section {
        display: flex;
        align-items: center;
        gap: 1rem;
    }

    .card-icon {
        width: 50px;
        height: 50px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 1.25rem;
    }

    .card-title {
        margin: 0;
        color: #2c3e50;
        font-size: 1.5rem;
        font-weight: 600;
    }

    .card-subtitle {
        margin: 0.25rem 0 0 0;
        color: #7f8c8d;
        font-size: 0.9rem;
    }

    .card-filters {
        display: flex;
        gap: 1rem;
    }

    .filter-item {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1rem;
        background: #f8faff;
        border: 1px solid #e8eeff;
        border-radius: 8px;
        color: #667eea;
        font-size: 0.9rem;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.3s ease;
    }

    .filter-item:hover {
        background: #667eea;
        color: white;
    }

    .card-body {
        padding: 0;
    }

    /* Table Styles */
    .applications-table {
        width: 100%;
        border-collapse: collapse;
        margin: 0;
    }

    .applications-table th {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        font-weight: 600;
        padding: 1.25rem 1rem;
        text-align: left;
        font-size: 0.85rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        border: none;
    }

    .applications-table td {
        padding: 1.25rem 1rem;
        border-bottom: 1px solid #f1f3f4;
        color: #2c3e50;
        font-size: 0.9rem;
        border: none;
    }

    .applications-table tr:last-child td {
        border-bottom: none;
    }

    .applications-table tr:hover {
        background: #f8faff;
        transform: translateY(-1px);
        box-shadow: 0 2px 8px rgba(102, 126, 234, 0.1);
        transition: all 0.2s ease;
    }

    /* Column Specific Styles */
    .serial-column {
        font-weight: 600;
        color: #667eea;
        text-align: center;
        width: 60px;
    }

    .user-cell {
        display: flex;
        align-items: center;
        gap: 0.75rem;
    }

    .user-avatar {
        width: 40px;
        height: 40px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-weight: 600;
        font-size: 0.9rem;
        flex-shrink: 0;
    }

    .user-info {
        display: flex;
        flex-direction: column;
        min-width: 0;
    }

    .user-name {
        font-weight: 600;
        color: #2c3e50;
        margin-bottom: 0.125rem;
    }

    .user-email {
        font-size: 0.8rem;
        color: #7f8c8d;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }

    .date-column {
        font-weight: 500;
        white-space: nowrap;
    }

    .category-cell {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        font-weight: 500;
        padding: 0.5rem 0.75rem;
        border-radius: 8px;
        width: fit-content;
    }

    .category-cell i {
        font-size: 0.9rem;
        width: 16px;
        text-align: center;
    }

    .category-cell.electricity {
        background: rgba(243, 156, 18, 0.1);
        color: #f39c12;
    }

    .category-cell.water {
        background: rgba(52, 152, 219, 0.1);
        color: #3498db;
    }

    .category-cell.gas {
        background: rgba(46, 204, 113, 0.1);
        color: #27ae60;
    }

    .category-cell.default {
        background: rgba(149, 165, 166, 0.1);
        color: #7f8c8d;
    }

    .readings-column {
        font-family: 'Courier New', monospace;
        font-weight: 600;
        text-align: right;
    }

    .readings-column.current {
        color: #27ae60;
        font-weight: 700;
    }

    .consumption-column {
        font-family: 'Courier New', monospace;
        font-weight: 600;
        text-align: right;
        color: #e67e22;
    }

    .amount-column {
        font-family: 'Courier New', monospace;
        font-weight: 700;
        text-align: right;
        color: #2c3e50;
        font-size: 0.95rem;
    }

    .status-column {
        text-align: center;
    }

    .created-column {
        color: #7f8c8d;
        font-size: 0.85rem;
        white-space: nowrap;
    }

    .actions-column {
        text-align: center;
        white-space: nowrap;
    }

    /* Status Badges */
    .status-badge {
        padding: 0.5rem 0.75rem;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        display: inline-block;
    }

    .status-badge.new {
        background: rgba(52, 152, 219, 0.1);
        color: #3498db;
        border: 1px solid rgba(52, 152, 219, 0.3);
    }

    .status-badge.processed {
        background: rgba(243, 156, 18, 0.1);
        color: #f39c12;
        border: 1px solid rgba(243, 156, 18, 0.3);
    }

    .status-badge.completed {
        background: rgba(46, 204, 113, 0.1);
        color: #27ae60;
        border: 1px solid rgba(46, 204, 113, 0.3);
    }

    .status-badge.rejected {
        background: rgba(231, 76, 60, 0.1);
        color: #c0392b;
        border: 1px solid rgba(231, 76, 60, 0.3);
    }

    /* Action Button */
    .btn-view {
        background: #667eea;
        color: white;
        border: none;
        padding: 0.5rem 1rem;
        border-radius: 8px;
        font-size: 0.8rem;
        font-weight: 500;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        transition: all 0.3s ease;
        cursor: pointer;
    }

    .btn-view:hover {
        background: #5a6fd8;
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
        text-decoration: none;
        color: white;
    }

    /* Empty State */
    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(20px);
        border-radius: 20px;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
        color: #7f8c8d;
    }

    .empty-icon {
        font-size: 4rem;
        margin-bottom: 1.5rem;
        opacity: 0.5;
    }

    .empty-state h3 {
        color: #2c3e50;
        margin-bottom: 1rem;
        font-size: 1.5rem;
    }

    .empty-state p {
        font-size: 1.1rem;
        margin-bottom: 0;
    }

    /* Pagination */
    .pagination {
        display: flex;
        justify-content: center;
        padding: 2rem;
        gap: 0.5rem;
    }

    .pagination li {
        list-style: none;
    }

    .pagination li a {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 40px;
        height: 40px;
        border-radius: 8px;
        background: white;
        color: #667eea;
        text-decoration: none;
        font-weight: 600;
        border: 1px solid #e8eeff;
        transition: all 0.3s ease;
    }

    .pagination li.active a {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-color: #667eea;
    }

    .pagination li a:hover {
        background: #667eea;
        color: white;
        transform: translateY(-1px);
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .table-responsive {
            overflow-x: auto;
        }

        .applications-table {
            min-width: 1000px;
        }
    }

    @media (max-width: 768px) {
        .admin-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 1rem;
        }

        .page-title {
            font-size: 1.75rem;
        }

        .card-header {
            flex-direction: column;
            gap: 1rem;
            align-items: flex-start;
        }

        .card-title-section {
            flex-direction: column;
            align-items: flex-start;
            text-align: left;
        }

        .applications-table th,
        .applications-table td {
            padding: 1rem 0.75rem;
            font-size: 0.8rem;
        }

        .empty-state {
            padding: 3rem 1rem;
        }

        .empty-icon {
            font-size: 3rem;
        }
    }

    /* Animations */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .admin-card {
        animation: fadeInUp 0.5s ease;
    }

    .applications-table tbody tr {
        animation: fadeInUp 0.3s ease forwards;
    }

    .applications-table tbody tr:nth-child(1) { animation-delay: 0.1s; }
    .applications-table tbody tr:nth-child(2) { animation-delay: 0.2s; }
    .applications-table tbody tr:nth-child(3) { animation-delay: 0.3s; }
    .applications-table tbody tr:nth-child(4) { animation-delay: 0.4s; }
    .applications-table tbody tr:nth-child(5) { animation-delay: 0.5s; }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Add hover effects to table rows
        const tableRows = document.querySelectorAll('.applications-table tbody tr');

        tableRows.forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
                this.style.boxShadow = '0 4px 12px rgba(102, 126, 234, 0.15)';
            });

            row.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
                this.style.boxShadow = 'none';
            });
        });

        // Add loading state for pagination
        const paginationLinks = document.querySelectorAll('.pagination a');

        paginationLinks.forEach(link => {
            link.addEventListener('click', function() {
                const table = document.querySelector('.applications-table');
                table.style.opacity = '0.6';
                table.style.pointerEvents = 'none';
            });
        });

        // Add filter functionality
        const filterItem = document.querySelector('.filter-item');
        if (filterItem) {
            filterItem.addEventListener('click', function() {
                this.classList.toggle('active');
                // Here you can add filter logic
            });
        }
    });
</script>